// 智能表单填写 - background.js
// 后台服务脚本：处理大模型调用和核心逻辑

// 统一的服务地址（如需调整直接修改此处）
const DEFAULT_BASE_SERVICE_URL = 'https://100resumepro.aitoolhub.top';
const DEFAULT_FREE_SERVICE_URL = DEFAULT_BASE_SERVICE_URL;
const DEFAULT_PAID_SERVICE_URL = DEFAULT_BASE_SERVICE_URL;

function getServiceEndpoints(userSettings = {}) {
    return {
        freeServiceUrl: userSettings.freeServiceUrl || DEFAULT_FREE_SERVICE_URL,
        paidServiceUrl: userSettings.paidServiceUrl || DEFAULT_PAID_SERVICE_URL
    };
}

function resolveSelectedService(userSettings = {}) {
    return userSettings.selectedService || userSettings.useMode || 'free';
}

function extractQuota(data) {
    if (!data) return undefined;
    return data.quota?.remainingQuota ?? data.remainingQuota ?? data.quota;
}

function normalizeAIContent(content) {
    if (!content || typeof content !== 'string') {
        throw new Error('AI 未返回有效内容');
    }
    let cleanContent = content.trim();
    if (cleanContent.startsWith('```json')) {
        cleanContent = cleanContent.slice(7);
    } else if (cleanContent.startsWith('```')) {
        cleanContent = cleanContent.slice(3);
    }
    if (cleanContent.endsWith('```')) {
        cleanContent = cleanContent.slice(0, -3);
    }
    return cleanContent.trim();
}

function parseAIContentToFillData(content) {
    const cleanContent = normalizeAIContent(content);
    const fillData = JSON.parse(cleanContent);
    return { cleanContent, fillData };
}

async function refreshPaidQuota(apiKey, userSettings = {}) {
    if (!apiKey) return;
    const { paidServiceUrl } = getServiceEndpoints(userSettings);

    try {
        const res = await fetch(`${paidServiceUrl}/api/activate`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ apiKey })
        });

        if (!res.ok) {
            return;
        }

        const data = await res.json().catch(() => ({}));
        const quota = extractQuota(data.data);
        if (quota !== undefined) {
            const updatedSettings = {
                ...userSettings,
                paidQuota: quota
            };
            await chrome.storage.local.set({ userSettings: updatedSettings });
            chrome.runtime.sendMessage({ action: 'refreshOptions' }).catch(() => {});
        }
    } catch (err) {
        // 忽略刷新失败
    }
}

async function callPaidAutoinput(prompt, userSettings = {}) {
    const apiKey = userSettings.activationKey;
    if (!apiKey) {
        throw new Error('付费密钥未配置');
    }

    const { paidServiceUrl } = getServiceEndpoints(userSettings);
    const res = await fetch(`${paidServiceUrl}/api/autoinput/paid`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ prompt, apiKey })
    });

    if (!res.ok) {
        const errorText = await res.text().catch(() => '');
        throw new Error(`付费服务调用失败: ${res.status} ${errorText}`.trim());
    }

    const data = await res.json().catch(() => ({}));
    if (!data.success) {
        throw new Error(data.error || data.message || '付费服务调用失败');
    }

    const content = data.data?.content || data.result || data.content || data.message;
    if (!content) {
        throw new Error('付费服务未返回内容');
    }

    const { cleanContent, fillData } = parseAIContentToFillData(content);
    console.log('清理后的内容:', cleanContent);
    console.log('解析后的fillData:', JSON.stringify(fillData, null, 2));

    const quota = extractQuota(data.data);
    if (quota !== undefined) {
        const updatedSettings = {
            ...userSettings,
            paidQuota: quota
        };
        await chrome.storage.local.set({ userSettings: updatedSettings });
        userSettings = updatedSettings;
    }
    await refreshPaidQuota(apiKey, userSettings);

    return fillData;
}

async function callFreeAutoinput(prompt, userSettings = {}) {
    const { freeServiceUrl } = getServiceEndpoints(userSettings);
    const res = await fetch(`${freeServiceUrl}/api/autoinput/free`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ prompt })
    });

    if (!res.ok) {
        const errorText = await res.text().catch(() => '');
        throw new Error(`免费服务调用失败: ${res.status} ${errorText}`.trim());
    }

    const data = await res.json().catch(() => ({}));
    if (!data.success) {
        throw new Error(data.error || data.message || '免费服务调用失败');
    }

    const content = data.data?.content || data.result || data.content || data.message;
    if (!content) {
        throw new Error('免费服务未返回内容');
    }

    const { cleanContent, fillData } = parseAIContentToFillData(content);
    console.log('清理后的内容:', cleanContent);
    console.log('解析后的fillData:', JSON.stringify(fillData, null, 2));
    return fillData;
}

// 扩展安装时的处理
chrome.runtime.onInstalled.addListener(function(details) {
    console.log('智能表单填写插件已安装');

    // 只在首次安装时设置默认配置
    if (details.reason === 'install') {
        console.log('首次安装，设置默认配置');

        // 检查是否已有配置，避免覆盖
        chrome.storage.local.get(['apiProvider'], function(result) {
            if (!result.apiProvider) {
                chrome.storage.local.set({
                    extensionEnabled: true,
                    apiProvider: 'custom',
                    apiKey: '',
                    apiModel: 'gpt-5.3-codex',
                    apiEndpoint: 'https://maas.hackerly.cn/v1/chat/completions',
                    userInfo: {
                        blocks: [
                            { id: 'default', title: '基础资料', content: '' }
                        ]
                    },
                    customRules: '',
                    fillHistory: []
                });
            }
        });
    } else if (details.reason === 'update') {
        console.log('插件更新');
    }
});

// ============================================
// 大模型API调用
// ============================================

/**
 * 调用大模型API生成表单填写内容
 */
async function callAIModel(formDescription, userInfo, config) {
    const { apiProvider, apiKey, apiModel, apiEndpoint, customRules, userSettings } = config;
    const selectedService = resolveSelectedService(userSettings);

    if (selectedService === 'paid' || selectedService === 'free') {
        const prompt = buildPrompt(formDescription, userInfo, customRules);
        if (selectedService === 'paid') {
            return await callPaidAutoinput(prompt, userSettings);
        }
        return await callFreeAutoinput(prompt, userSettings);
    }

    if (selectedService === 'local') {
        const localKey = userSettings?.localApiKey;
        const localEndpoint = userSettings?.apiUrl || 'https://api.deepseek.com/chat/completions';
        const localModel = userSettings?.apiModel || 'deepseek-chat';

        if (!localKey) {
            throw new Error('请先配置本地 API Key');
        }

        const prompt = buildPrompt(formDescription, userInfo, customRules);
        const response = await fetch(localEndpoint, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${localKey}`
            },
            body: JSON.stringify({
                model: localModel,
                messages: [
                    { role: 'system', content: '你是一个智能填表助手。根据用户信息和表单字段的标签，生成合适的填写内容。必须返回纯JSON格式，不要使用markdown代码块标记（如```json或```）。如果字段提供了autoinputId，请务必在结果中返回autoinputId并用它来定位字段。' },
                    { role: 'user', content: prompt }
                ],
                temperature: 0.7,
                max_tokens: 4096
            })
        });

        if (!response.ok) {
            const errorText = await response.text().catch(() => '');
            throw new Error(`本地模型调用失败: ${response.status} ${errorText}`.trim());
        }

        const data = await response.json();
        const content = data?.choices?.[0]?.message?.content;
        if (!content) {
            throw new Error('本地模型未返回有效内容');
        }

        console.log('AI原始返回:', content);
        const { cleanContent, fillData } = parseAIContentToFillData(content);
        console.log('清理后的内容:', cleanContent);
        console.log('解析后的fillData:', JSON.stringify(fillData, null, 2));
        return fillData;
    }

    if (!apiKey) {
        throw new Error('请先配置 API 密钥');
    }
    if (!apiEndpoint) {
        throw new Error('请先配置 API 端点');
    }
    if (!apiModel) {
        throw new Error('请先配置模型');
    }

    console.log(`调用模型: provider=${apiProvider}, model=${apiModel}, endpoint=${apiEndpoint}`);

    // 构建提示词
    const prompt = buildPrompt(formDescription, userInfo, customRules);

    try {
        let requestBody, headers, endpoint;

        if (apiProvider === 'zhipu' || apiProvider === 'siliconflow' || apiProvider === 'deepseek' || apiProvider === 'openai' || apiProvider === 'custom') {
            // OpenAI 格式 API（智谱、硅基流动、DeepSeek都兼容此格式）
            endpoint = apiEndpoint;
            headers = {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            };
            requestBody = {
                model: apiModel,
                messages: [
                    {
                        role: 'system',
                        content: '你是一个智能填表助手。根据用户信息和表单字段的标签，生成合适的填写内容。必须返回纯JSON格式，不要使用markdown代码块标记（如```json或```）。如果字段提供了autoinputId，请务必在结果中返回autoinputId并用它来定位字段。'
                    },
                    {
                        role: 'user',
                        content: prompt
                    }
                ],
                temperature: 0.7,
                max_tokens: 4096
            };

            // 智谱AI特有的参数
            if (apiProvider === 'zhipu') {
                requestBody.stream = false;
                requestBody.top_p = 0.7;
            }

            // DeepSeek 特有参数
            if (apiProvider === 'deepseek') {
                requestBody.frequency_penalty = 0;
                requestBody.presence_penalty = 0;
            }
        } else if (apiProvider === 'claude') {
            // Claude API
            endpoint = apiEndpoint;
            headers = {
                'Content-Type': 'application/json',
                'x-api-key': apiKey,
                'anthropic-version': '2023-06-01'
            };
            requestBody = {
                model: apiModel,
                messages: [
                    {
                        role: 'user',
                        content: prompt
                    }
                ],
                max_tokens: 4096,
                system: '你是一个测试助手，我会发给你网页的表单字段信息。根据用户信息和表单字段的标签，生成合适的测试填写内容。内容尽量模拟真实情况，禁止出现**或者某某等等，返回格式必须是严格的JSON对象。如果字段提供了autoinputId，请务必在结果中返回autoinputId并用它来定位字段。'
            };
        }

        console.log('AI请求:', JSON.stringify({
            provider: apiProvider,
            endpoint: endpoint,
            model: apiModel,
            requestBody: requestBody
        }, null, 2));

        const response = await fetch(endpoint, {
            method: 'POST',
            headers: headers,
            body: JSON.stringify(requestBody)
        });

        if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`API 调用失败: ${response.status} - ${errorText}`);
        }

        const data = await response.json();
        console.log('AI响应:', JSON.stringify(data, null, 2));

        // 解析响应
        let content;
        if (apiProvider === 'zhipu' || apiProvider === 'siliconflow' || apiProvider === 'deepseek' || apiProvider === 'openai' || apiProvider === 'custom') {
            // OpenAI 格式响应
            content = data.choices[0].message.content;
        } else if (apiProvider === 'claude') {
            // Claude 格式响应
            content = data.content[0].text;
        }

        console.log('AI原始返回:', content);
        const { cleanContent, fillData } = parseAIContentToFillData(content);
        console.log('清理后的内容:', cleanContent);
        console.log('解析后的fillData:', JSON.stringify(fillData, null, 2));

        // 保存到历史记录
        chrome.storage.local.get(['fillHistory'], function(result) {
            const history = result.fillHistory || [];
            history.unshift({
                url: formDescription.pageInfo.url,
                title: formDescription.pageInfo.title,
                timestamp: Date.now(),
                fieldCount: fillData.fields?.length || 0
            });
            // 只保留最近20条记录
            if (history.length > 20) {
                history.pop();
            }
            chrome.storage.local.set({ fillHistory: history });
        });

        return fillData;

    } catch (error) {
        console.error('调用大模型失败:', error);
        throw error;
    }
}

/**
 * 构建AI提示词
 */
function buildPrompt(formDescription, userInfo, customRules) {
    let prompt = `请根据以下用户信息和表单字段，生成填写内容。\n\n`;

    // 添加用户信息
    if (userInfo) {
        const blocks = Array.isArray(userInfo.blocks) ? userInfo.blocks : [];
        if (blocks.length > 0) {
            prompt += `【用户资料】\n`;
            blocks.forEach(block => {
                const title = block.title ? block.title.trim() : '资料';
                const content = block.content ? block.content.trim() : '';
                if (!content) return;
                prompt += `- ${title}: ${content}\n`;
            });
            prompt += `\n`;
        }
    }

    // 添加自定义规则
    if (customRules) {
        prompt += `【自定义规则】\n${customRules}\n\n`;
    }

    // 添加表单信息
    prompt += `【表单信息】\n`;
    prompt += `页面标题: ${formDescription.pageInfo.title}\n`;
    prompt += `页面URL: ${formDescription.pageInfo.url}\n\n`;

    prompt += `【需要填写的字段】\n`;
    formDescription.fields.forEach((field, index) => {
        const locatorInfo = [];
        if (field.id) locatorInfo.push(`id=${field.id}`);
        if (field.name) locatorInfo.push(`name=${field.name}`);
        if (field.autoinputId) locatorInfo.push(`autoinputId=${field.autoinputId}`);
        const locatorText = locatorInfo.length > 0 ? `，定位: ${locatorInfo.join(', ')}` : '';

        prompt += `${index + 1}. 类型: ${field.type}, 标签: ${field.label}${locatorText}`;
        if (field.placeholder) prompt += `, 占位符: ${field.placeholder}`;
        if (field.required) prompt += ` (必填)`;
        if (field.options && field.options.length > 0) {
            const optionDesc = field.options.map(o => {
                const parts = [];
                if (o.value) parts.push(`value=${o.value}`);
                if (o.text || o.label) parts.push(`label=${o.text || o.label}`);
                if (o.autoinputId) parts.push(`autoinputId=${o.autoinputId}`);
                return `{${parts.join(', ')}}`;
            }).join('; ');
            prompt += `\n   可选项: ${optionDesc}`;
        }
        prompt += `\n`;
    });

    prompt += `\n请以JSON格式返回填写结果，格式如下：
{
  "fields": [
    {
      "id": "字段ID（如果有）",
      "name": "字段name（如果有）",
      "autoinputId": "字段的autoinputId（如果提供）",
      "value": "填写的内容",
      "type": "text|select|radio|checkbox|date|textarea"
    }
  ]
}

重要提示：
1. 直接返回纯JSON格式，不要使用markdown代码块（不要包含\`\`\`json或\`\`\`标记）
2. 如果提供了 autoinputId，务必在结果中返回 autoinputId，并用它定位字段；其次使用 id；再其次使用 name；不要自行创建新的 selector
3. 没有 id/name/autoinputId 的字段不要返回
4. 对于 select 类型，value 必须是选项列表中的 value
5. 对于 radio/checkbox 类型，value 应该是对应选项的 value
6. 日期格式使用 YYYY-MM-DD
7. 即使用户信息为空，也要为所有字段生成合理的测试值，不要返回空 fields`;

    return prompt;
}

// ============================================
// 消息处理
// ============================================

chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
    console.log('收到消息:', request);

    switch(request.action) {
        case 'getStorage':
            // 获取存储数据
            chrome.storage.local.get(request.keys, function(result) {
                sendResponse(result);
            });
            return true;

        case 'setStorage':
            // 设置存储数据
            chrome.storage.local.set(request.data, function() {
                sendResponse({ success: true });
            });
            return true;

        case 'identifyAndFill':
            // 识别表单并调用AI填写
            handleIdentifyAndFill(request, sender, sendResponse);
            return true;

        case 'testAPI':
            // 测试API连接
            testAPIConnection(request.config).then(result => {
                sendResponse(result);
            }).catch(error => {
                sendResponse({ success: false, error: error.message });
            });
            return true;

        case 'testConnection':
            testLocalAPIConnection(request).then(result => {
                sendResponse(result);
            }).catch(error => {
                sendResponse({ success: false, error: error.message });
            });
            return true;

        default:
            sendResponse({ error: 'Unknown action' });
    }
    return true;
});

/**
 * 处理识别和填写流程
 */
async function handleIdentifyAndFill(request, sender, sendResponse) {
    try {
        console.log('========== Background 开始处理 ==========');
        const { formDescription, tabId } = request;
        console.log('接收到的表单描述:', JSON.stringify(formDescription, null, 2));
        console.log('目标tabId:', tabId);

        // 获取配置和用户信息
        chrome.storage.local.get(
            ['apiProvider', 'apiKey', 'apiModel', 'apiEndpoint', 'userInfo', 'customRules', 'userSettings'],
            async function(config) {
                console.log('加载的配置:', config);
                try {
                    const selectedService = resolveSelectedService(config.userSettings || {});
                    if (selectedService !== 'free' && selectedService !== 'paid' && selectedService !== 'local') {
                        if (!config.apiKey || !config.apiEndpoint || !config.apiModel) {
                            const miss = [];
                            if (!config.apiKey) miss.push('API 密钥');
                            if (!config.apiEndpoint) miss.push('API 端点');
                            if (!config.apiModel) miss.push('模型');
                            throw new Error(`配置不完整：${miss.join('、')}`);
                        }
                    }

                    console.log('开始调用AI模型...');
                    // 调用大模型
                    const fillData = await callAIModel(formDescription, config.userInfo, config);
                    console.log('AI返回的fillData:', JSON.stringify(fillData, null, 2));

                    console.log('发送填写数据到content script, tabId:', tabId);
                    // 发送填写数据到content script
                    chrome.tabs.sendMessage(tabId, {
                        action: 'fillForm',
                        data: fillData
                    });

                    sendResponse({
                        success: true,
                        fillData: fillData,
                        meta: {
                            provider: selectedService === 'free' || selectedService === 'paid'
                                ? selectedService
                                : config.apiProvider,
                            model: selectedService === 'free' || selectedService === 'paid'
                                ? undefined
                                : config.apiModel,
                            endpoint: selectedService === 'free' || selectedService === 'paid'
                                ? undefined
                                : config.apiEndpoint,
                            fieldCount: fillData?.fields?.length || 0
                        }
                    });
                    console.log('========== Background 处理完成 ==========\n');

                } catch (error) {
                    console.error('AI调用失败:', error);
                    sendResponse({
                        success: false,
                        error: error.message
                    });
                }
            }
        );

    } catch (error) {
        console.error('处理失败:', error);
        sendResponse({
            success: false,
            error: error.message
        });
    }
}

async function testLocalAPIConnection({ apiKey, apiUrl, apiModel }) {
    if (!apiKey) {
        throw new Error('API 密钥未配置');
    }

    const endpoint = apiUrl || 'https://api.deepseek.com/chat/completions';
    const model = apiModel || 'deepseek-chat';
    const testPrompt = '请回复：连接成功。';

    const response = await fetch(endpoint, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${apiKey}`
        },
        body: JSON.stringify({
            model: model,
            messages: [
                { role: 'user', content: testPrompt }
            ],
            max_tokens: 50,
            temperature: 0
        })
    });

    if (!response.ok) {
        const errorText = await response.text().catch(() => '');
        throw new Error(`API 返回错误: ${response.status} ${errorText}`.trim());
    }

    const data = await response.json().catch(() => null);
    const content = data?.choices?.[0]?.message?.content;
    if (!content) {
        throw new Error('API 未返回有效内容');
    }

    return { success: true, message: 'API 连接成功', response: content };
}

/**
 * 测试API连接
 */
async function testAPIConnection(config) {
    try {
        const testPrompt = '你好，这是一个测试消息。请回复"连接成功"。';
        const { apiProvider, apiKey, apiModel, apiEndpoint } = config;

        if (!apiKey) {
            throw new Error('API 密钥未配置');
        }

        let headers, requestBody;

        if (apiProvider === 'zhipu' || apiProvider === 'siliconflow' || apiProvider === 'deepseek' || apiProvider === 'openai' || apiProvider === 'custom') {
            headers = {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            };
            requestBody = {
                model: apiModel,
                messages: [
                    { role: 'user', content: testPrompt }
                ],
                max_tokens: 100,
                temperature: 0.7
            };

            // 智谱AI特有参数
            if (apiProvider === 'zhipu') {
                requestBody.stream = false;
            }
        } else if (apiProvider === 'claude') {
            headers = {
                'Content-Type': 'application/json',
                'x-api-key': apiKey,
                'anthropic-version': '2023-06-01'
            };
            requestBody = {
                model: apiModel,
                messages: [
                    { role: 'user', content: testPrompt }
                ],
                max_tokens: 100
            };
        }

        const response = await fetch(apiEndpoint, {
            method: 'POST',
            headers: headers,
            body: JSON.stringify(requestBody)
        });

        if (!response.ok) {
            throw new Error(`API 返回错误: ${response.status}`);
        }

        return { success: true, message: 'API 连接成功' };

    } catch (error) {
        console.error('API 测试失败:', error);
        throw error;
    }
}

// 标签页更新监听
chrome.tabs.onUpdated.addListener(function(tabId, changeInfo, tab) {
    if (changeInfo.status === 'complete') {
        console.log('页面加载完成:', tab.url);
    }
});
