// 智能表单填写 - contentScript.js
// 这个脚本会在网页中运行，用于识别和填写表单

console.log('智能表单填写插件已加载');

let autoinputIdCounter = 0;

function getOrCreateAutoinputId(element) {
    if (!element.dataset.autoinputId) {
        autoinputIdCounter += 1;
        element.dataset.autoinputId = `autoinput-${Date.now()}-${autoinputIdCounter}`;
    }
    return element.dataset.autoinputId;
}

function safeCssEscape(value) {
    if (value === undefined || value === null) {
        return '';
    }
    const str = String(value);
    if (typeof CSS !== 'undefined' && CSS.escape) {
        return CSS.escape(str);
    }
    return str.replace(/"/g, '\\"');
}

let lastIdentifiedForm = null;

// 注入高亮样式
function injectStyles() {
    const style = document.createElement('style');
    style.textContent = `
        .autoinput-highlight {
            outline: 3px solid #3182ce !important;
            outline-offset: 2px;
            transition: all 0.3s ease;
        }
        .autoinput-filled {
            outline: 3px solid #48bb78 !important;
            outline-offset: 2px;
            transition: all 0.3s ease;
        }
        .autoinput-tooltip {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #2d3748;
            color: white;
            padding: 15px 20px;
            border-radius: 8px;
            z-index: 999999;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            max-width: 300px;
            font-family: Arial, sans-serif;
        }
    `;
    document.head.appendChild(style);
}

// ============================================
// 表单识别模块
// ============================================

/**
 * 识别页面上的所有表单元素
 */
function identifyFormElements() {
    const formElements = {
        textInputs: [],
        radioGroups: [],
        checkboxGroups: [],
        selects: [],
        dateInputs: [],
        textareas: [],
        buttons: []
    };

    // 识别文本输入框（包括 text, email, tel, number 等）
    const textTypes = ['text', 'email', 'tel', 'number', 'url', 'search', 'password'];
    textTypes.forEach(type => {
        const inputs = document.querySelectorAll(`input[type="${type}"]`);
        inputs.forEach(input => {
            formElements.textInputs.push(extractFieldInfo(input, 'text'));
        });
    });

    // 识别单选按钮组
    const radioInputs = document.querySelectorAll('input[type="radio"]');
    const radioGroups = {};
    radioInputs.forEach(radio => {
        const name = radio.name;
        if (!radioGroups[name]) {
            radioGroups[name] = [];
        }
        radioGroups[name].push(extractFieldInfo(radio, 'radio'));
    });
    formElements.radioGroups = Object.values(radioGroups);

    // 识别多选框
    const checkboxInputs = document.querySelectorAll('input[type="checkbox"]');
    const checkboxGroups = {};
    checkboxInputs.forEach(checkbox => {
        const name = checkbox.name;
        if (!checkboxGroups[name]) {
            checkboxGroups[name] = [];
        }
        checkboxGroups[name].push(extractFieldInfo(checkbox, 'checkbox'));
    });
    formElements.checkboxGroups = Object.values(checkboxGroups);

    // 识别下拉菜单
    const selects = document.querySelectorAll('select');
    selects.forEach(select => {
        formElements.selects.push(extractFieldInfo(select, 'select'));
    });

    // 识别日期输入
    const dateInputs = document.querySelectorAll('input[type="date"], input[type="datetime-local"], input[type="time"]');
    dateInputs.forEach(input => {
        formElements.dateInputs.push(extractFieldInfo(input, 'date'));
    });

    // 识别文本域
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        formElements.textareas.push(extractFieldInfo(textarea, 'textarea'));
    });

    // 识别提交按钮
    const submitButtons = document.querySelectorAll('button[type="submit"], input[type="submit"], button:not([type])');
    submitButtons.forEach(button => {
        formElements.buttons.push({
            tag: button.tagName,
            text: button.textContent.trim(),
            type: button.type || 'button',
            id: button.id
        });
    });

    return formElements;
}

/**
 * 提取字段信息
 */
function extractFieldInfo(element, fieldType) {
    const info = {
        type: fieldType,
        id: element.id,
        name: element.name,
        element: element, // 保存元素引用以便后续操作
        autoinputId: getOrCreateAutoinputId(element),
        options: []
    };

    // 提取标签信息
    info.label = extractLabel(element);

    // 提取占位符
    if (element.placeholder) {
        info.placeholder = element.placeholder;
    }

    // 提取必填标记
    if (element.required) {
        info.required = true;
    }

    // 针对不同类型的额外信息
    if (fieldType === 'select') {
        const options = element.querySelectorAll('option');
        info.options = Array.from(options).map(opt => ({
            value: opt.value,
            text: opt.textContent.trim()
        }));
    } else if (fieldType === 'radio' || fieldType === 'checkbox') {
        info.value = element.value;
        info.checked = element.checked;
        const label = element.closest('label');
        if (label) {
            info.optionLabel = label.textContent.trim();
        }
    }

    return info;
}

/**
 * 提取字段的标签文本
 */
function extractLabel(element) {
    // 方法1：通过 for 属性关联的 label
    if (element.id) {
        const label = document.querySelector(`label[for="${element.id}"]`);
        if (label) {
            return label.textContent.trim();
        }
    }

    // 方法2：父级 label
    const parentLabel = element.closest('label');
    if (parentLabel) {
        const text = parentLabel.textContent.trim();
        // 移除元素本身的文本（如果有）
        return text.replace(element.value || '', '').trim();
    }

    // 方法3：查找之前的兄弟元素或父元素
    let parent = element.parentElement;
    let attempts = 0;
    while (parent && attempts < 3) {
        const prevSibling = parent.previousElementSibling;
        if (prevSibling && (prevSibling.tagName === 'LABEL' || prevSibling.classList.contains('label'))) {
            return prevSibling.textContent.trim();
        }
        parent = parent.parentElement;
        attempts++;
    }

    // 方法4：使用 placeholder 或 name
    return element.placeholder || element.name || element.id || '未命名字段';
}

/**
 * 高亮显示识别到的表单元素
 */
function highlightFormElements(formElements) {
    // 清除之前的高亮
    document.querySelectorAll('.autoinput-highlight').forEach(el => {
        el.classList.remove('autoinput-highlight');
    });

    let totalFields = 0;

    // 高亮文本输入
    formElements.textInputs.forEach(field => {
        if (field.element) {
            field.element.classList.add('autoinput-highlight');
            totalFields++;
        }
    });

    // 高亮下拉菜单
    formElements.selects.forEach(field => {
        if (field.element) {
            field.element.classList.add('autoinput-highlight');
            totalFields++;
        }
    });

    // 高亮日期输入
    formElements.dateInputs.forEach(field => {
        if (field.element) {
            field.element.classList.add('autoinput-highlight');
            totalFields++;
        }
    });

    // 高亮文本域
    formElements.textareas.forEach(field => {
        if (field.element) {
            field.element.classList.add('autoinput-highlight');
            totalFields++;
        }
    });

    // 高亮单选按钮组
    formElements.radioGroups.forEach(group => {
        group.forEach(field => {
            if (field.element) {
                field.element.parentElement?.classList.add('autoinput-highlight');
            }
        });
        totalFields++;
    });

    // 高亮多选框组
    formElements.checkboxGroups.forEach(group => {
        group.forEach(field => {
            if (field.element) {
                field.element.parentElement?.classList.add('autoinput-highlight');
            }
        });
        totalFields++;
    });

    return totalFields;
}

/**
 * 显示提示信息
 */
function showTooltip(message, duration = 3000) {
    // 移除旧的提示
    const oldTooltip = document.querySelector('.autoinput-tooltip');
    if (oldTooltip) {
        oldTooltip.remove();
    }

    const tooltip = document.createElement('div');
    tooltip.className = 'autoinput-tooltip';
    tooltip.textContent = message;
    document.body.appendChild(tooltip);

    setTimeout(() => {
        tooltip.remove();
    }, duration);
}

function clearAllHighlights() {
    document.querySelectorAll('.autoinput-highlight, .autoinput-filled').forEach(el => {
        el.classList.remove('autoinput-highlight', 'autoinput-filled');
    });
}


// ============================================
// 消息监听
// ============================================

// 监听来自background的消息
chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
    console.log('收到消息:', request);

    switch(request.action) {
        case 'identifyForm':
            // 识别表单
            console.log('========== 开始识别表单 ==========');
            const formElements = identifyFormElements();
            console.log('识别到的表单元素:', JSON.stringify(formElements, null, 2));
            const fieldCount = highlightFormElements(formElements);
            console.log('高亮的字段数量:', fieldCount);

            // 构建简化的表单描述用于发送给background
            const simplifiedForm = {
                pageInfo: {
                    title: document.title,
                    url: window.location.href
                },
                fields: []
            };

            // 整合所有字段信息
            formElements.textInputs.forEach(f => {
                const desc = createFieldDescription(f);
                console.log('文本输入字段:', desc);
                simplifiedForm.fields.push(desc);
            });
            formElements.selects.forEach(f => {
                const desc = createFieldDescription(f);
                console.log('下拉菜单字段:', desc);
                simplifiedForm.fields.push(desc);
            });
            formElements.dateInputs.forEach(f => {
                const desc = createFieldDescription(f);
                console.log('日期字段:', desc);
                simplifiedForm.fields.push(desc);
            });
            formElements.textareas.forEach(f => {
                const desc = createFieldDescription(f);
                console.log('文本域字段:', desc);
                simplifiedForm.fields.push(desc);
            });
    formElements.radioGroups.forEach(group => {
        const radioField = {
            type: 'radio',
            label: group[0]?.label || '单选组',
            name: group[0]?.name,
            options: group.map(g => ({
                value: g.value,
                label: g.optionLabel,
                autoinputId: g.autoinputId
            }))
        };
        console.log('单选组字段:', radioField);
        simplifiedForm.fields.push(radioField);
    });
    formElements.checkboxGroups.forEach(group => {
        const checkboxField = {
            type: 'checkbox',
            label: group[0]?.label || '多选组',
            name: group[0]?.name,
            options: group.map(g => ({
                value: g.value,
                label: g.optionLabel,
                autoinputId: g.autoinputId
            }))
        };
        console.log('多选组字段:', checkboxField);
        simplifiedForm.fields.push(checkboxField);
            });

            console.log('简化的表单描述:', JSON.stringify(simplifiedForm, null, 2));
            console.log('========== 表单识别完成 ==========\n');

            lastIdentifiedForm = simplifiedForm;

            showTooltip(`识别到 ${fieldCount} 个表单字段`);
            sendResponse({ success: true, formElements: simplifiedForm, fieldCount: fieldCount });
            return true;

        case 'fillForm':
            // 填写表单
            console.log('========== 收到填写请求 ==========');
            console.log('request.data:', JSON.stringify(request.data, null, 2));
            fillFormData(request.data);
            clearAllHighlights();
            sendResponse({ success: true });
            break;

        case 'clearHighlight':
            // 清除高亮
            clearAllHighlights();
            sendResponse({ success: true });
            break;

        default:
            sendResponse({ error: 'Unknown action' });
    }
    return true;
});

/**
 * 创建字段描述（用于发送给AI）
 */
function createFieldDescription(field) {
    const desc = {
        type: field.type,
        label: field.label
    };

    // 保存选择器信息（重要！）
    if (field.id) {
        desc.id = field.id;
    }
    if (field.name) {
        desc.name = field.name;
    }
    if (field.autoinputId) {
        desc.autoinputId = field.autoinputId;
    }

    if (field.placeholder) desc.placeholder = field.placeholder;
    if (field.required) desc.required = true;
    if (field.options && field.options.length > 0) {
        desc.options = field.options;
    }

    return desc;
}

/**
 * 填写表单数据
 */
function fillFormData(fillData) {
    console.log('========== 开始填写表单 ==========');
    console.log('接收到的fillData:', JSON.stringify(fillData, null, 2));

    if (!fillData || !fillData.fields) {
        console.error('没有可填写的数据');
        showTooltip('没有可填写的数据');
        return;
    }

    console.log('字段总数:', fillData.fields.length);
    let filledCount = 0;
    let failedCount = 0;

    fillData.fields.forEach((fieldData, index) => {
        console.log(`\n--- 处理字段 ${index + 1} ---`);
        console.log('字段数据:', JSON.stringify(fieldData, null, 2));

        // 兼容新旧格式
        const selector = fieldData.selector || {};
        let fieldId = fieldData.id || selector.id;
        let fieldName = fieldData.name || selector.name;
        let fieldAutoId = fieldData.autoinputId || selector.autoinputId;
        const value = fieldData.value;
        const fieldType = fieldData.type;

        console.log('fieldId:', fieldId);
        console.log('fieldName:', fieldName);
        console.log('fieldAutoId:', fieldAutoId);
        console.log('value:', value);
        console.log('fieldType:', fieldType);

        if (!fieldAutoId && lastIdentifiedForm && Array.isArray(lastIdentifiedForm.fields)) {
            const matchCandidates = lastIdentifiedForm.fields.filter(f => {
                return (fieldData.label && f.label === fieldData.label) || (fieldData.name && f.label === fieldData.name);
            });
            if (matchCandidates.length === 1) {
                const matched = matchCandidates[0];
                if (!fieldAutoId && matched.autoinputId) fieldAutoId = matched.autoinputId;
                if (!fieldId && matched.id) fieldId = matched.id;
                if (!fieldName && matched.name) fieldName = matched.name;
                console.log('通过已识别字段匹配到定位信息:', matched);
            }
        }

        if (!fieldId && !fieldName && !fieldAutoId) {
            console.error('字段缺少id、name或autoinputId，跳过此字段');
            failedCount++;
            return;
        }

        try {
            let element;

            // 根据选择器查找元素
            console.log('开始查找元素...');

            if (fieldId) {
                console.log(`尝试通过ID查找: #${fieldId}`);
                element = document.querySelector(`#${safeCssEscape(fieldId)}`);
                console.log('查找结果:', element);
            }

            if (!element && fieldAutoId) {
                const selectorStr = `[data-autoinput-id="${safeCssEscape(fieldAutoId)}"]`;
                console.log(`尝试通过autoinputId查找: ${selectorStr}`);
                element = document.querySelector(selectorStr);
                console.log('查找结果:', element);
            }

            if (!element && fieldName) {
                console.log(`尝试通过name查找: [name="${fieldName}"]`);

                // 对于单选和多选框，需要特殊处理
                if (fieldType === 'radio' || fieldType === 'checkbox') {
                    console.log('这是单选或多选框，需要匹配value');
                    console.log(`完整选择器: input[name="${fieldName}"][value="${value}"]`);
                    element = document.querySelector(`input[name="${safeCssEscape(fieldName)}"][value="${safeCssEscape(value)}"]`);
                    console.log('查找结果:', element);
                    if (element) {
                        element.checked = true;
                        console.log('✓ 成功设置单选/多选框');
                    }
                } else {
                    // 普通输入框、select、textarea等
                    element = document.querySelector(`[name="${safeCssEscape(fieldName)}"]`);
                    console.log('查找结果:', element);
                }
            }

            if (!element) {
                console.error('✗ 未找到元素');
                failedCount++;
                return;
            }

            console.log('✓ 成功找到元素');
            console.log('元素标签:', element.tagName);
            console.log('元素类型:', element.type);
            console.log('元素ID:', element.id);
            console.log('元素name:', element.name);

            // 根据元素类型填写值
            const elementType = element.tagName.toLowerCase();
            const inputType = element.type?.toLowerCase();

            console.log('elementType:', elementType);
            console.log('inputType:', inputType);

            if (elementType === 'input') {
                if (inputType === 'radio' || inputType === 'checkbox') {
                    const groupName = fieldName || element.name;
                    if (groupName && groupName.trim() !== '') {
                        const checkedInGroup = document.querySelector(`input[type="${inputType}"][name="${safeCssEscape(groupName)}"]:checked`);
                        if (checkedInGroup) {
                            console.log('该组已有选择，跳过覆盖');
                            return;
                        }
                    }
                    if (element.checked) {
                        console.log('已存在选中值，跳过覆盖');
                        return;
                    }
                    console.log('设置checked属性为:', value);
                    element.checked = value === undefined ? true : Boolean(value);
                } else if (inputType === 'date' || inputType === 'datetime-local' || inputType === 'time') {
                    if (element.value) {
                        console.log('已有日期值，跳过覆盖');
                        return;
                    }
                    console.log('设置日期值为:', value);
                    element.value = value;
                } else {
                    if (element.value && element.value.trim() !== '') {
                        console.log('已有输入值，跳过覆盖');
                        return;
                    }
                    console.log('设置input值为:', value);
                    element.value = value;
                }
                filledCount++;
                console.log('✓ 成功填写input字段');
            } else if (elementType === 'select') {
                if (element.value && element.value !== '') {
                    console.log('已有选择值，跳过覆盖');
                    return;
                }
                console.log('设置select值为:', value);
                element.value = value;
                filledCount++;
                console.log('✓ 成功填写select字段');
            } else if (elementType === 'textarea') {
                if (element.value && element.value.trim() !== '') {
                    console.log('已有文本内容，跳过覆盖');
                    return;
                }
                console.log('设置textarea值为:', value);
                element.value = value;
                filledCount++;
                console.log('✓ 成功填写textarea字段');
            }

            // 触发 change 和 input 事件
            console.log('触发事件...');
            const events = ['input', 'change', 'blur'];
            events.forEach(eventType => {
                const event = new Event(eventType, { bubbles: true });
                element.dispatchEvent(event);
            });

            // 添加填写完成的高亮
            element.classList.remove('autoinput-highlight');
            element.classList.add('autoinput-filled');

            console.log('✓ 字段填写完成');

        } catch (error) {
            console.error('✗ 填写字段失败:', error);
            console.error('错误堆栈:', error.stack);
            console.error('字段数据:', fieldData);
            failedCount++;
        }
    });

    console.log('\n========== 填写完成 ==========');
    console.log('成功:', filledCount, '失败:', failedCount);
    showTooltip(`已填写 ${filledCount} 个字段`);
}

// 页面加载完成后初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initContentScript);
} else {
    initContentScript();
}

function initContentScript() {
    injectStyles();
    console.log('智能表单填写插件已初始化');
}
