// 智能填表助手 - options.js

// 统一的接口地址（直接在此修改）
const DEFAULT_BASE_SERVICE_URL = 'https://100resumepro.aitoolhub.top';
const DEFAULT_FREE_SERVICE_URL = DEFAULT_BASE_SERVICE_URL;
const DEFAULT_PAID_SERVICE_URL = DEFAULT_BASE_SERVICE_URL;

// 默认配置
const DEFAULT_CONFIG = {
    userSettings: {
        selectedService: 'free',
        activationKey: '',
        paidQuota: 0,
        useMode: 'local',
        localApiKey: '',
        apiUrl: 'https://api.deepseek.com/chat/completions',
        apiModel: 'deepseek-chat',
        freeServiceUrl: DEFAULT_FREE_SERVICE_URL,
        paidServiceUrl: DEFAULT_PAID_SERVICE_URL
    },
    userInfo: {
        blocks: [
            { id: 'default', title: '基础资料', content: '' }
        ]
    },
    customRules: '',
    fillHistory: []
};

let elements = {};
let userSettings = { ...DEFAULT_CONFIG.userSettings };
let paymentPollingTimer = null;
let currentPaymentModal = null;
let activationKeySaveTimer = null;

// 初始化
document.addEventListener('DOMContentLoaded', function() {
    loadConfig();
    initEventListeners();
    loadHistory();
    registerRuntimeListeners();
});

/**
 * 初始化事件监听器
 */
function initEventListeners() {
    // 标签切换
    document.querySelectorAll('.tab').forEach(tab => {
        tab.addEventListener('click', function() {
            switchTab(this.dataset.tab);
        });
    });

    // 模式切换
    document.querySelectorAll('[data-action^="select-"]').forEach(btn => {
        btn.addEventListener('click', function() {
            const mode = this.dataset.action.replace('select-', '');
            selectMode(mode);
        });
    });

    // 保存按钮
    document.getElementById('saveBtn').addEventListener('click', saveConfig);

    // 重置按钮
    document.getElementById('resetBtn').addEventListener('click', resetConfig);

    // 导出按钮
    document.getElementById('exportBtn').addEventListener('click', exportConfig);

    // 导入按钮
    document.getElementById('importBtn').addEventListener('click', function() {
        document.getElementById('importFile').click();
    });

    // 文件导入
    document.getElementById('importFile').addEventListener('change', importConfig);

    // 模式相关按钮
    const btnSelectFree = document.getElementById('btnSelectFree');
    if (btnSelectFree) {
        btnSelectFree.addEventListener('click', () => selectService('free'));
    }

    const activatePaidServiceBtn = document.getElementById('activatePaidServiceBtn');
    if (activatePaidServiceBtn) {
        activatePaidServiceBtn.addEventListener('click', () => handleActivatePaidClick());
    }

    const purchasePaidServiceBtn = document.getElementById('purchasePaidServiceBtn');
    if (purchasePaidServiceBtn) {
        purchasePaidServiceBtn.addEventListener('click', () => handlePurchasePaidPlan());
    }

    const testLocalBtn = document.getElementById('testLocalBtn');
    if (testLocalBtn) {
        testLocalBtn.addEventListener('click', () => testLocalConnection());
    }

    const saveLocalBtn = document.getElementById('saveLocalBtn');
    if (saveLocalBtn) {
        saveLocalBtn.addEventListener('click', () => saveLocalConfig());
    }

    const paidActivationKey = document.getElementById('paidActivationKey');
    if (paidActivationKey) {
        paidActivationKey.addEventListener('input', (e) => handleActivationKeyInput(e.target.value));
    }

    document.querySelectorAll('.model-card').forEach(card => {
        card.addEventListener('click', (e) => {
            if (e.target.tagName === 'BUTTON' || e.target.tagName === 'INPUT' || e.target.tagName === 'A') {
                return;
            }
            const mode = card.dataset.mode;
            if (mode) {
                selectService(mode);
            }
        });
    });

    // 新增资料块
    document.getElementById('addProfileBtn').addEventListener('click', function() {
        addProfileBlock();
    });
}

/**
 * 切换标签
 */
function switchTab(tabName) {
    // 更新标签状态
    document.querySelectorAll('.tab').forEach(tab => {
        tab.classList.remove('active');
        if (tab.dataset.tab === tabName) {
            tab.classList.add('active');
        }
    });

    // 更新内容显示
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.remove('active');
    });
    document.getElementById(tabName).classList.add('active');
}

/**
 * 加载配置
 */
function loadConfig() {
    chrome.storage.local.get(Object.keys(DEFAULT_CONFIG), function(result) {
        console.log('从存储加载的配置:', result);

        // 合并默认配置和已保存配置
        const config = { ...DEFAULT_CONFIG, ...result };
        console.log('合并后的配置:', config);

        const settings = config.userSettings || DEFAULT_CONFIG.userSettings;
        userSettings = { ...DEFAULT_CONFIG.userSettings, ...settings };
        cacheModelElements();
        initializeModelMode();

        // 填充用户信息
        if (config.userInfo) {
            const mergedUserInfo = normalizeUserInfo(config.userInfo);
            renderProfileBlocks(mergedUserInfo.blocks);
        } else {
            renderProfileBlocks(DEFAULT_CONFIG.userInfo.blocks);
        }

        // 填充自定义规则
        document.getElementById('customRules').value = config.customRules || '';

        // 更新 API 预设（只在endpoint和model为空时才更新）
    });
}

// 选择服务
function selectService(mode) {
    if (mode === 'paid' && !userSettings.activationKey) {
        showStatus('请先激活付费密钥后再选择付费模式', 'warning');
        return;
    }
    // 移除所有卡片的选中状态
    elements.serviceCards.forEach(card => {
        card.classList.remove('selected');
    });

    // 选中当前卡片
    const selectedCard = document.querySelector(`[data-mode="${mode}"]`);
    if (selectedCard) {
        selectedCard.classList.add('selected');
    }

    // 更新按钮状态
    if (elements.btnSelectFree) {
        if (mode === 'free') {
            elements.btnSelectFree.textContent = '当前正在使用';
            elements.btnSelectFree.disabled = true;
        } else {
            elements.btnSelectFree.textContent = '切换到免费版';
            elements.btnSelectFree.disabled = false;
        }
    }

    // 保存选择
    userSettings.selectedService = mode;
    chrome.storage.local.set({ userSettings: userSettings });
}

// 设置当前选中的服务
function selectCurrentService() {
    const currentMode = userSettings.selectedService || 'free';
    selectService(currentMode);
}


// 更新免费版状态（简化版 - 无计费逻辑）
function updateFreeStatus() {
    // 免费版无使用限制，只显示状态
    if (elements.freeCurrentService) {
        elements.freeCurrentService.textContent = '免费版';
    }
    if (elements.freeRemainingQuota) {
        elements.freeRemainingQuota.textContent = '无限制';
    }
    // 移除重置时间显示逻辑
}

// 更新付费版状态（简化版）
function updatePaidStatus() {
    const paidQuota = userSettings.paidQuota || 0;

    if (paidQuota > 0) {
        elements.paidRemainingQuota.textContent = `${paidQuota} 次`;
        elements.activatePaidServiceBtn.textContent = '刷新';
        elements.activatePaidServiceBtn.disabled = false;
    } else {
        elements.paidRemainingQuota.textContent = '0 次';
        elements.activatePaidServiceBtn.textContent = '刷新';
        elements.activatePaidServiceBtn.disabled = false;
    }

    // 同步激活密钥输入框，方便用户随时查看/修改
    if (elements.paidActivationKey) {
        elements.paidActivationKey.value = userSettings.activationKey || '';
    }
}

// 显示付费版激活区域
function showPaidActivationSection() {
    elements.paidActivationSection.style.display = 'block';
    elements.paidActivationKey.value = userSettings.activationKey || '';
    elements.paidActivationKey.focus();
}

// 隐藏付费版激活区域
function hidePaidActivationSection() {
    // 保持输入区域可见，仅重置为当前已保存的密钥
    elements.paidActivationSection.style.display = 'block';
    elements.paidActivationKey.value = userSettings.activationKey || '';
}

function getPaidServiceUrl() {
    return DEFAULT_PAID_SERVICE_URL;
}

function getFreeServiceUrl() {
    return DEFAULT_FREE_SERVICE_URL;
}

function showResumePrompt() {
    alert('未找到简历内容，请先在「简历管理」标签下填写或同步简历。');
}

async function normalizeServiceUrls(settings) {
    // 直接使用代码中的默认地址，不再依赖缓存或外部配置
    return {
        ...settings,
        freeServiceUrl: DEFAULT_FREE_SERVICE_URL,
        paidServiceUrl: DEFAULT_PAID_SERVICE_URL
    };
}

function handleActivationKeyInput(value) {
    const trimmed = value.trim();
    if (activationKeySaveTimer) {
        clearTimeout(activationKeySaveTimer);
    }
    activationKeySaveTimer = setTimeout(async () => {
        const updatedSettings = {
            ...userSettings,
            activationKey: trimmed
        };
        userSettings = updatedSettings;
        await chrome.storage.local.set({ userSettings: updatedSettings });
    }, 300);
}

async function handleActivatePaidClick() {
    const inputKey = elements.paidActivationKey?.value.trim();
    const activationKey = inputKey || userSettings.activationKey;
    if (!activationKey) {
        showStatus('请输入激活密钥', 'error');
        return;
    }
    await reactivatePaidService(activationKey);
}

async function reactivatePaidService(activationKey) {
    try {
        elements.activatePaidServiceBtn.disabled = true;
        elements.activatePaidServiceBtn.textContent = '同步中...';
        showStatus('正在同步付费配额...', 'loading');

        const result = await validateActivationKey(activationKey);
        clearLoadingStatus();

        if (result.success) {
            showStatus('已刷新付费配额', 'success');
        } else {
            showStatus(result.message || '未查询到剩余配额，请确认密钥是否正确', 'error');
        }
    } catch (error) {
        // console.error('重新激活失败:', error);
        clearLoadingStatus();
        showStatus('重新激活失败：' + error.message, 'error');
    } finally {
        updatePaidStatus();
    }
}

// 购买付费版（创建订单并轮询支付状态）
async function handlePurchasePaidPlan() {
    const paidServiceUrl = getPaidServiceUrl();
    try {
        showStatus('正在创建支付订单...', 'loading');

        const createRes = await fetch(`${paidServiceUrl}/api/pay/create`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ planType: 'basic' })
        });

        clearLoadingStatus();

        if (!createRes.ok) {
            const errText = await createRes.text();
            // console.error('[支付] 创建订单响应文本:', errText);
            throw new Error(`创建订单失败：${createRes.status} ${createRes.statusText} - ${errText}`);
        }

        const createData = await createRes.json();
        if (!createData.success) {
            throw new Error(createData.error || '创建订单失败');
        }

        const { orderId, qrCode, originalPrice, payPrice, currency } = createData.data || {};
        if (!orderId || !qrCode) {
            throw new Error('订单数据不完整，缺少 orderId 或二维码');
        }

        showPaymentModal({ orderId, qrCode, originalPrice, payPrice, currency });
        startPaymentPolling(orderId);
        showStatus('订单创建成功，请扫码完成支付', 'success');
    } catch (error) {
        // console.error('创建支付订单失败:', error);
        clearLoadingStatus();
        showStatus(error.message || '创建订单失败，请稍后重试', 'error');
    }
}

function showPaymentModal({ orderId, qrCode, originalPrice, payPrice, currency = 'CNY' }) {
    closePaymentModal();

    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay payment-modal';
    overlay.style.cssText = `
        position: fixed;
        inset: 0;
        background: rgba(0,0,0,0.45);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 2147483647;
        opacity: 1;
        visibility: visible;
    `;

    const modal = document.createElement('div');
    modal.className = 'modal';
    modal.style.cssText = `
        width: 460px;
        background: #fff;
        border-radius: 16px;
        padding: 24px;
        box-shadow: 0 20px 50px rgba(0,0,0,0.16);
        z-index: 2147483647;
        transform: scale(1) translateY(0);
    `;

    modal.innerHTML = `
        <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:12px;">
            <h3 style="margin:0; font-size:18px; display:flex; align-items:center; gap:8px;">
                <span style="display:inline-block; width:8px; height:8px; border-radius:50%; background:#7c3aed;"></span>
                支付宝扫码支付
            </h3>
            <button id="closePaymentModal" style="border:none; background:transparent; cursor:pointer; font-size:16px;">✕</button>
        </div>
        <p style="margin:0 0 12px; color:#555;">订单号：${orderId}</p>
        <div style="display:flex; gap:16px; align-items:center;">
            <div style="display:flex; flex-direction:column; align-items:center; gap:8px;">
                <img src="${qrCode}" alt="支付二维码" style="width:180px; height:180px; border:1px solid #eee; border-radius:12px; object-fit:contain; box-shadow:0 6px 20px rgba(0,0,0,0.08);">
                <span style="font-size:13px; color:#444;">仅支持支付宝扫码</span>
            </div>
            <div style="flex:1;">

                <div style="margin:8px 0 12px; padding:12px; border-radius:10px; background:linear-gradient(135deg,#f5f3ff,#ede9fe); color:#4c1d95; font-weight:600;">
                    支付成功即获得 500 次使用额度
                </div>
                <p style="margin:0 0 4px; color:#444;">原价：${originalPrice ?? '--'} ${currency}</p>
                <p style="margin:0 0 12px; color:#444;">现价：<strong style="font-size:18px; color:#7c3aed;">${payPrice ?? '--'} ${currency}</strong></p>
                <p style="margin:0; color:#666; font-size:12px; line-height:1.5;">
                    支付完成后将生成激活密钥，请复制并在下方“确认激活”区域手动激活。
                </p>
            </div>
        </div>
    `;

    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    overlay.classList.add('active');
    currentPaymentModal = overlay;

    // 关闭事件
    const closeBtn = modal.querySelector('#closePaymentModal');
    closeBtn.addEventListener('click', () => {
        closePaymentModal();
    });
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay) {
            closePaymentModal();
        }
    });
}

function closePaymentModal() {
    if (currentPaymentModal && document.body.contains(currentPaymentModal)) {
        document.body.removeChild(currentPaymentModal);
    }
    currentPaymentModal = null;
    stopPaymentPolling();
}

function showActivationKeyModal(apiKey, orderId) {
    // 关闭已有弹窗
    if (currentPaymentModal && document.body.contains(currentPaymentModal)) {
        document.body.removeChild(currentPaymentModal);
    }

    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay activation-modal';
    overlay.style.cssText = `
        position: fixed;
        inset: 0;
        background: rgba(0,0,0,0.45);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 2147483647;
        opacity: 1;
        visibility: visible;
    `;

    const modal = document.createElement('div');
    modal.className = 'modal';
    modal.style.cssText = `
        width: 440px;
        background: #fff;
        border-radius: 16px;
        padding: 24px;
        box-shadow: 0 20px 50px rgba(0,0,0,0.16);
        z-index: 2147483647;
        transform: scale(1) translateY(0);
    `;

    const maskedKey = `${apiKey.substring(0, 4)}****${apiKey.substring(apiKey.length - 4)}`;

    modal.innerHTML = `
        <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:12px;">
            <h3 style="margin:0; font-size:18px; display:flex; align-items:center; gap:8px;">
                <span style="display:inline-block; width:8px; height:8px; border-radius:50%; background:#10b981;"></span>
                支付成功，复制激活密钥
            </h3>
            <button id="closeActivationModal" style="border:none; background:transparent; cursor:pointer; font-size:16px;">✕</button>
        </div>
        ${orderId ? `<p style="margin:0 0 8px; color:#666;">订单号：${orderId}</p>` : ''}
        <div style="padding:14px; background:#f8fafc; border:1px dashed #cbd5e1; border-radius:10px; display:flex; gap:10px; align-items:center; margin-bottom:12px;">
            <code id="activationKeyText" style="flex:1; word-break:break-all; font-size:14px; color:#111; background:transparent;">${apiKey}</code>
            <button id="copyActivationKey" style="padding:8px 12px; border:none; background:#7c3aed; color:#fff; border-radius:8px; cursor:pointer;">复制</button>
        </div>
        <p style="margin:0 0 6px; color:#111; font-weight:600;">激活说明</p>
        <ul style="margin:0 0 12px 16px; padding:0; color:#444; line-height:1.6; font-size:13px;">
            <li>使用上方“激活密钥”手动填写后再点击“确认激活”。</li>
            <li>激活后立即获得 500 次使用额度。</li>
            <li>请妥善保管密钥，避免泄露。</li>
        </ul>
        <p style="margin:0; color:#6b7280; font-size:12px;">密钥示例：${maskedKey}</p>
    `;

    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    currentPaymentModal = overlay;

    const closeBtn = modal.querySelector('#closeActivationModal');
    closeBtn.addEventListener('click', () => closePaymentModal());
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay) {
            closePaymentModal();
        }
    });

    const copyBtn = modal.querySelector('#copyActivationKey');
    const keyText = modal.querySelector('#activationKeyText');
    copyBtn.addEventListener('click', async () => {
        try {
            await navigator.clipboard.writeText(apiKey);
            showStatus('激活密钥已复制，请前往手动激活', 'success');
        } catch (err) {
            // console.error('复制激活密钥失败:', err);
            showStatus('复制失败，请手动选择复制', 'error');
        }
    });
}

function startPaymentPolling(orderId) {
    const paidServiceUrl = getPaidServiceUrl();
    stopPaymentPolling();

    paymentPollingTimer = setInterval(async () => {
        // 如果弹窗已被关闭，停止轮询
        if (!currentPaymentModal || !document.body.contains(currentPaymentModal)) {
            stopPaymentPolling();
            return;
        }
        try {
            const res = await fetch(`${paidServiceUrl}/api/pay/status?orderId=${encodeURIComponent(orderId)}`);
            if (!res.ok) {
                // console.warn('支付状态查询失败:', res.status, res.statusText);
                return;
            }
            const data = await res.json();
            if (!data.success) {
                // console.warn('支付状态查询返回失败:', data.error || data.code);
                return;
            }

            const status = data.data?.status;
            if (status === 'paid') {
                stopPaymentPolling();
                closePaymentModal();

                const apiKey = data.data?.apiKey;
                if (apiKey) {
                    showActivationKeyModal(apiKey, orderId);
                    showStatus('支付成功，已生成激活码，请复制后手动激活', 'success');
                } else {
                    showStatus('支付成功，但未获得激活密钥，请联系客服', 'warning');
                    // console.warn('[支付] 支付成功但未返回激活密钥:', data);
                }
            }
        } catch (err) {
            // console.warn('支付状态查询异常:', err);
        }
    }, 3000);
}

function stopPaymentPolling() {
    if (paymentPollingTimer) {
        clearInterval(paymentPollingTimer);
        paymentPollingTimer = null;
    }
}

// 确认付费版激活
async function confirmPaidActivation() {
    const activationKey = elements.paidActivationKey.value.trim();

    if (!activationKey) {
        showStatus('请输入激活密钥', 'error');
        return;
    }

    try {
        // 验证激活密钥
        const result = await validateActivationKey(activationKey);

        if (result.success) {
            hidePaidActivationSection();
            updatePaidStatus();
            selectService('paid'); // 自动选择付费版
            const quotaText = (result.remainingQuota ?? userSettings.paidQuota) ? `，剩余${result.remainingQuota ?? userSettings.paidQuota}次` : '';
            showStatus(`付费版激活成功${quotaText}`, 'success');
        } else {
            showStatus(result.message || '激活密钥无效', 'error');
        }
    } catch (error) {
        // console.error('Activation failed:', error);
        showStatus('激活失败：' + error.message, 'error');
    }
}

// 加载本地配置
function loadLocalConfig() {
    const savedApiKey = userSettings.localApiKey || '';
    const savedApiUrl = userSettings.apiUrl || 'https://api.deepseek.com/chat/completions';
    const savedApiModel = userSettings.apiModel || 'deepseek-chat';

    // 确保元素存在后再设置值
    if (elements && elements.localApiKeyInput) {
        elements.localApiKeyInput.value = savedApiKey;
    } else {
        // console.error('❌ Cannot find localApiKeyInput element!');
    }

    // 设置API地址
    if (elements && elements.apiUrlInput) {
        elements.apiUrlInput.value = savedApiUrl;
    } else {
        // console.error('❌ Cannot find apiUrlInput element!');
    }

    // 设置模型名称
    if (elements && elements.apiModelInput) {
        elements.apiModelInput.value = savedApiModel;
    } else {
        // console.error('❌ Cannot find apiModelInput element!');
    }
}

// 保存本地配置（更新版本）
async function saveLocalConfig() {
    const apiKey = elements.localApiKeyInput.value;
    const apiUrl = elements.apiUrlInput.value || 'https://api.deepseek.com/chat/completions';
    const apiModel = elements.apiModelInput.value || 'deepseek-chat';

    if (!apiKey) {
        showStatus('请输入API密钥', 'error');
        return;
    }

    const config = {
        ...userSettings,
        useMode: 'local',
        apiUrl: apiUrl,
        apiModel: apiModel,
        localApiKey: apiKey,
        selectedService: 'local'
    };

    try {
        await chrome.storage.local.set({ userSettings: config });
        userSettings = config;

        // 确保选择模式为本地
        selectService('local');

        // 验证保存是否成功
        const verifyResult = await chrome.storage.local.get(['userSettings']);

        showStatus('本地配置已保存', 'success');
    } catch (error) {
        // console.error('Error saving local config:', error);
        showStatus('保存失败：' + error.message, 'error');
    }
}

// 测试连接（更新版本）
async function testLocalConnection() {
    const apiKey = elements.localApiKeyInput.value;
    const apiUrl = elements.apiUrlInput.value || 'https://api.deepseek.com/chat/completions';
    const apiModel = elements.apiModelInput.value || 'deepseek-chat';

    if (!apiKey) {
        showStatus('请先输入DeepSeek API密钥', 'error');
        return;
    }

    elements.testLocalBtn.disabled = true;
    elements.testLocalBtn.textContent = '测试中...';

    try {
        // 发送测试请求到background script
        const response = await chrome.runtime.sendMessage({
            action: 'testConnection',
            apiKey: apiKey,
            apiUrl: apiUrl,
            apiModel: apiModel
        });

        if (response.success) {
            showStatus('DeepSeek API连接测试成功！', 'success');
        } else {
            throw new Error(response.error);
        }
    } catch (error) {
        // console.error('Connection test failed:', error);
        showStatus('连接测试失败：' + error.message, 'error');
    } finally {
        elements.testLocalBtn.disabled = false;
        elements.testLocalBtn.textContent = '测试连接';
    }
}




// 验证激活密钥（模拟）
async function validateActivationKey(key) {
    try {
        const paidServiceUrl = getPaidServiceUrl();

        const res = await fetch(`${paidServiceUrl}/api/activate`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ apiKey: key })
        });
        const data = await res.json().catch(() => ({}));

        if (!res.ok || data.success === false) {
            throw new Error(data.error || data.message || `激活接口错误: ${res.status}`);
        }

        const remainingQuota = data.data?.quota?.remainingQuota
            ?? data.data?.remainingQuota
            ?? data.data?.quota;

        if (remainingQuota === undefined || remainingQuota === null) {
            return { success: false, message: '未查询到剩余配额，请确认密钥是否正确' };
        }

        // 成功则更新本地设置
        const config = {
            ...userSettings,
            activationKey: key,
            paidQuota: remainingQuota,
            selectedService: 'paid'
        };
        await chrome.storage.local.set({ userSettings: config });
        userSettings = config;
        updatePaidStatus();
        selectService('paid');

        return { success: true, remainingQuota };
    } catch (error) {
        // console.error('激活密钥验证失败:', error);
        return { success: false, message: error.message };
    }
}



function clearLoadingStatus() {
    const loadingStatuses = document.querySelectorAll('.status-toast.loading');
    loadingStatuses.forEach(status => {
        if (status._isLoading) {
            const parent = status.parentElement;
            if (parent && parent.contains(status)) {
                parent.removeChild(status);
            }
        }
    });

    // 如果容器已空则移除容器
    const container = document.getElementById('status-toast-container');
    if (container && container.childElementCount === 0 && document.body.contains(container)) {
        document.body.removeChild(container);
    }
}




// 生成岗位话术（岗位卡片中）
async function generateJobIntroduction(jobUrl, job) {
    try {

    // 检查简历是否存在
    if (!userResume || userResume.trim().length === 0) {
        showStatus('请先在简历管理中同步简历内容', 'error');
        showResumePrompt();
        return;
    }

        // 验证job对象是否有效
        if (!job || !job.jobName) {
            showStatus('岗位信息不完整，无法AI 打招呼', 'error');
            // console.error('❌ Invalid job object:', job);
            return;
        }

        showStatus('正在AI 打招呼...', 'loading');

        // 准备岗位描述
        let jobDetailText = '';
        if (job.jobDetail && job.jobDetail.jobDescription) {
            jobDetailText = job.jobDetail.jobDescription;
        }

        // 构建岗位信息
        const jobInfo = `
岗位名称：${job.jobName}
公司名称：${job.jobCompany}
薪资待遇：${job.jobSalary || '薪资面议'}
工作地点：${job.location || '未知'}

岗位详情：
${jobDetailText || '暂无详情'}
        `.trim();

        // 发送请求到background script
        const response = await chrome.runtime.sendMessage({
            action: 'generateIntroduction',
            jobInfo: jobInfo
        });

        if (response && response.success) {
            clearLoadingStatus(); // 清除loading状态

            // 在岗位详情弹窗内部更新内容
            const generatedAt = new Date().toISOString();
            updateIntroductionDisplay(jobUrl, response.introduction, generatedAt);

            showStatus('话术生成成功！', 'success');

            // 直接更新当前jobHistory中的数据，避免重新加载
            if (jobHistory && jobHistory[jobUrl]) {
                jobHistory[jobUrl].selfIntroduction = response.introduction;
                jobHistory[jobUrl].introductionGeneratedAt = generatedAt;
            }

            // 更新岗位卡片中的话术（后台保存）
            await updateJobIntroduction(jobUrl, response.introduction);

            // 自动复制到剪贴板
            try {
                await copyIntroductionToClipboard(response.introduction);
                showStatus('话术已自动复制到剪贴板！', 'success');
            } catch (error) {
                // console.warn('自动复制失败，但生成成功:', error);
                showStatus('话术生成成功！（自动复制失败）', 'warning');
            }

        } else {
            throw new Error(response.error || '生成失败');
        }

    } catch (error) {
        // console.error('❌ AI 打招呼失败:', error);
        clearLoadingStatus(); // 清除loading状态
        showStatus('生成失败：' + error.message, 'error');
    }
}

// 更新岗位的话术
async function updateJobIntroduction(jobUrl, introduction) {
    try {
        const result = await chrome.storage.local.get(['jobHistory']);
        const jobHistory = result.jobHistory || {};

        if (jobHistory[jobUrl]) {
            jobHistory[jobUrl].selfIntroduction = introduction;
            jobHistory[jobUrl].introductionGeneratedAt = new Date().toISOString();
            await chrome.storage.local.set({ jobHistory });
        }
    } catch (error) {
        // console.error('保存话术失败:', error);
    }
}

// 显示话术弹窗
function showJobIntroductionModal(job, introduction) {
    // 移除现有弹窗
    const existingModal = document.querySelector('.job-introduction-modal');
    if (existingModal) {
        document.body.removeChild(existingModal);
    }

    // 创建新弹窗
    const modal = document.createElement('div');
    modal.className = 'modal-overlay job-introduction-modal';
    modal.innerHTML = `
        <div class="modal">
            <div class="modal-header">
                <h3 class="modal-title">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                    </svg>
                    ${job.jobName} - 沟通话术
                </h3>
                <button class="modal-close">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                        <line x1="18" y1="6" x2="6" y2="18" stroke="currentColor" stroke-width="2"
                            stroke-linecap="round" />
                        <line x1="6" y1="6" x2="18" y2="18" stroke="currentColor" stroke-width="2"
                            stroke-linecap="round" />
                    </svg>
                </button>
            </div>

            <div class="modal-body">
                <div class="introduction-content">
                    <div class="introduction-text">${parseMarkdown(introduction)}</div>
                    <div class="introduction-actions">
                        <button class="btn btn-secondary" id="copyJobIntroductionBtn">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect>
                                <path d="M5 15H4a2 2 0 01-2-2V4a2 2 0 012-2h9a2 2 0 012 2v1"></path>
                            </svg>
                            复制话术
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    document.body.appendChild(modal);

    // 绑定关闭事件
    modal.querySelector('.modal-close').addEventListener('click', () => {
        document.body.removeChild(modal);
    });

    // 点击遮罩层关闭
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            document.body.removeChild(modal);
        }
    });

    // 绑定复制事件
    const copyBtn = modal.querySelector('#copyJobIntroductionBtn');
    if (copyBtn) {
        copyBtn.addEventListener('click', () => {
            navigator.clipboard.writeText(introduction).then(() => {
                showStatus('话术已复制到剪贴板', 'success');
            }).catch((error) => {
                // console.error('复制失败:', error);
                showStatus('复制失败', 'error');
            });
        });
    }
}

// 生成AI自我介绍（岗位详情弹窗中）
async function generateSelfIntroduction(jobUrl, modal) {
    const jobId = jobUrl.replace(/[^a-zA-Z0-9]/g, '');
    const contentEl = document.getElementById(`introduction-${jobId}`);
    const loadingEl = document.getElementById(`loading-${jobId}`);

    try {
        // 显示加载状态
        if (contentEl && loadingEl) {
            contentEl.style.display = 'none';
            loadingEl.style.display = 'flex';
        }

        // 获取岗位信息
        const job = jobHistory.find(j => j.jobUrl === jobUrl);
    if (!job) {
        throw new Error('未找到岗位信息');
    }

    // 检查用户简历
    if (!userResume || userResume.trim() === '') {
        showResumePrompt();
        throw new Error('请先在简历管理中完善您的简历信息');
    }

        // 构建岗位信息
        const jobInfo = `
岗位名称：${job.jobName}
公司名称：${job.jobCompany}
工作描述：${job.jobDetail?.jobDescription || '暂无详情'}
工作经验要求：${job.experience || '未明确'}
学历要求：${job.education || '未明确'}
        `;

        // 发送生成请求到background script（统一走后台逻辑）
        const response = await chrome.runtime.sendMessage({
            action: 'generateIntroduction',
            jobInfo: jobInfo
        });

        if (response && response.success) {
            // 保存生成的自我介绍
            const updatedJob = {
                ...job,
                selfIntroduction: response.introduction,
                introductionGeneratedAt: new Date().toISOString()
            };

            // 更新存储
            const historyResult = await chrome.storage.local.get(['jobHistory']);
            const jobHistory = historyResult.jobHistory || {};
            jobHistory[jobUrl] = updatedJob;
            await chrome.storage.local.set({ jobHistory });

            // 更新本地数据
            const jobIndex = jobHistory.findIndex(j => j.jobUrl === jobUrl);
            if (jobIndex !== -1) {
                jobHistory[jobIndex] = updatedJob;
            }

            // 更新显示
            updateIntroductionDisplay(jobUrl, response.introduction, updatedJob.introductionGeneratedAt);

            showStatus('自我介绍生成成功！', 'success');

            // 自动复制到剪贴板
            try {
                await copyIntroductionToClipboard(response.introduction);
                showStatus('自我介绍已自动复制到剪贴板！', 'success');
            } catch (error) {
                // console.warn('自动复制失败，但生成成功:', error);
                showStatus('自我介绍生成成功！（自动复制失败）', 'warning');
            }
        } else {
            throw new Error(response.error || '生成失败');
        }

    } catch (error) {
        // console.error('生成自我介绍失败:', error);
        showStatus('生成失败：' + error.message, 'error');

        // 恢复显示 - 添加null检查
        if (contentEl && loadingEl) {
            contentEl.style.display = 'flex';
            loadingEl.style.display = 'none';
        }
    }
}

// 更新自我介绍显示
function updateIntroductionDisplay(jobUrl, introduction, generatedAt) {
    const jobId = jobUrl.replace(/[^a-zA-Z0-9]/g, '');

    // 尝试在岗位详情弹窗中找到自我介绍区域
    const modal = document.querySelector('.job-detail-modal');
    if (!modal) {
        // console.warn('❌ 未找到岗位详情弹窗');
        return;
    }

    // 查找加载状态元素
    const loadingEl = modal.querySelector(`#loading-${jobId}`);

    // 查找或创建内容区域
    let contentEl = modal.querySelector(`#introduction-${jobId}`);

    if (!contentEl) {
        // 如果没有找到ID元素，说明是在有评分的情况下，需要查找现有的自我介绍区域
        const generatorCard = modal.querySelector('.generator-card');
        if (generatorCard) {
            // 查找现有的介绍内容区域
            contentEl = generatorCard.querySelector('.introduction-content');

            if (!contentEl) {
                // 如果还没有介绍内容区域，创建一个
                const generatorDesc = generatorCard.querySelector('.generator-desc');
                const generateBtn = generatorCard.querySelector('.btn-generate[data-url]');

                if (generatorDesc && generateBtn) {
                    // 替换描述区域为内容区域
                    contentEl = document.createElement('div');
                    contentEl.className = 'introduction-content';
                    contentEl.id = `introduction-${jobId}`;

                    // 隐藏生成按钮和描述
                    generatorDesc.style.display = 'none';
                    generateBtn.style.display = 'none';

                    // 插入内容区域
                    generatorCard.insertBefore(contentEl, generatorDesc.nextSibling);
                }
            }
        }
    }

    if (contentEl) {
        contentEl.innerHTML = `
            <div class="introduction-content" style="margin-top: 0;">
                <div class="introduction-text">
                    ${parseMarkdown(introduction)}
                </div>
                <div class="introduction-actions">
                    <button class="btn-generate btn-copy-intro" data-url="${jobUrl}">
                        📋 复制话术
                    </button>
                    <button class="btn-generate btn-regenerate-intro" data-url="${jobUrl}">
                        🔄 重新生成
                    </button>
                </div>
            </div>
        `;
        contentEl.style.display = 'flex';

        // 绑定事件
        const copyBtn = contentEl.querySelector('.btn-copy-intro');
        if (copyBtn) {
            copyBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                copyIntroduction(jobUrl);
            });
        }

        const regenerateBtn = contentEl.querySelector('.btn-regenerate-intro');
        if (regenerateBtn) {
            regenerateBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                regenerateIntroduction(jobUrl);
            });
        }
    } else {
        // console.error('❌ 无法找到或创建内容区域');
    }

    // 隐藏加载状态
    if (loadingEl) {
        loadingEl.style.display = 'none';
    }
}

// 复制自我介绍到剪贴板
async function copyIntroduction(jobUrl) {
    const job = jobHistory.find(j => j.jobUrl === jobUrl);
    if (job && job.selfIntroduction) {
        try {
            await copyIntroductionToClipboard(job.selfIntroduction);
            showStatus('自我介绍已复制到剪贴板', 'success');
        } catch (error) {
            // console.error('复制失败:', error);
            showStatus('复制失败', 'error');
        }
    }
}

// 通用复制方法：复制自我介绍内容到剪贴板
async function copyIntroductionToClipboard(introduction) {
    if (!introduction) {
        throw new Error('没有可复制的内容');
    }

    // 清理HTML标签，保留纯文本
    const cleanText = introduction.replace(/<[^>]*>/g, '').trim();

    if (!cleanText) {
        throw new Error('复制内容为空');
    }

    try {
        // 使用现代的clipboard API
        await navigator.clipboard.writeText(cleanText);
    } catch (error) {
        // 如果clipboard API失败，尝试传统方法
        try {
            const textArea = document.createElement('textarea');
            textArea.value = cleanText;
            textArea.style.position = 'fixed';
            textArea.style.left = '-999999px';
            textArea.style.top = '-999999px';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            document.execCommand('copy');
            document.body.removeChild(textArea);
        } catch (fallbackError) {
            throw new Error('复制失败: ' + fallbackError.message);
        }
    }
}

// 重新生成自我介绍
async function regenerateIntroduction(jobUrl) {
    if (confirm('确定要重新生成自我介绍吗？这将消耗一次AI使用次数。')) {
        const modal = document.querySelector('.job-detail-modal');
        if (!modal) {
            showStatus('请在岗位详情弹窗中使用此功能', 'error');
            // console.error('❌ Modal not found for regeneration');
            return;
        }
        generateSelfIntroduction(jobUrl, modal);
    }
}

async function deleteJob(jobUrl, jobName) {
    if (!confirm(`确定要删除"${jobName}"这条历史记录吗？\n\n删除后将无法恢复，且会清除相关的AI评分数据。`)) {
        return;
    }

    try {

        // 从历史记录中删除
        const result = await chrome.storage.local.get(['jobHistory']);
        const jobHistory = result.jobHistory || {};
        delete jobHistory[jobUrl];
        await chrome.storage.local.set({ jobHistory });

        // 从评分记录中删除
        const scoreResult = await chrome.storage.local.get(['scoredJobs']);
        const scoredJobs = scoreResult.scoredJobs || {};
        delete scoredJobs[jobUrl];
        await chrome.storage.local.set({ scoredJobs });

        // 更新显示
        await loadJobHistory();
        filterAndDisplayJobs();
        updateStatistics();
        showStatus(`已删除"${jobName}"的历史记录`, 'success');

    } catch (error) {
        // console.error('删除历史记录失败:', error);
        showStatus('删除失败: ' + error.message, 'error');
    }
}

async function clearAllHistory() {
    const recordCount = jobHistory?.length || 0;

    if (recordCount === 0) {
        showStatus('暂无历史记录可清空', 'warning');
        return;
    }

    if (!confirm(`确定要清空所有历史记录吗？\n\n当前共有 ${recordCount} 条记录，清空后将无法恢复！\n此操作将同时删除所有AI评分数据。`)) {
        return;
    }

    try {

        // 清空所有历史记录
        await chrome.storage.local.set({
            jobHistory: {},
            scoredJobs: {}
        });

        // 更新显示
        await loadJobHistory();
        filterAndDisplayJobs();
        updateStatistics();
        showStatus(`已清空所有历史记录（共 ${recordCount} 条）`, 'success');

    } catch (error) {
        // console.error('清空历史记录失败:', error);
        showStatus('清空失败: ' + error.message, 'error');
    }
}

async function saveResume() {
    const resumeContent = elements.resumeContent.value;

    try {
        await chrome.storage.local.set({ userResume: resumeContent });
        userResume = resumeContent;
        showStatus('简历已保存', 'success');
    } catch (error) {
        // console.error('Error saving resume:', error);
        showStatus('保存失败：' + error.message, 'error');
    }
}

function resetResume() {
    if (confirm('确定要重置简历内容吗？')) {
        elements.resumeContent.value = '';
        saveResume();
    }
}

function autoSaveResume() {
    // 防抖自动保存
    clearTimeout(autoSaveTimer);
    autoSaveTimer = setTimeout(() => {
        saveResume();
    }, 2000);
}

async function exportData() {
    try {
        const data = await chrome.storage.local.get(null);
        const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = '100ResumePro-backup.json';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        showStatus('数据已导出为 JSON 文件', 'success');
    } catch (error) {
        showStatus('导出失败：' + error.message, 'error');
    }
}

async function importData(file) {
    try {
        const text = await file.text();
        const parsed = JSON.parse(text);
        await chrome.storage.local.set(parsed);
        showStatus('数据已导入，请刷新页面查看最新信息', 'success');
        await loadData();
    } catch (error) {
        showStatus('导入失败：' + error.message, 'error');
    }
}

function setupStorageListener() {
    // 监听存储变化，自动刷新历史记录
    storageListener = (changes, namespace) => {
        if (namespace === 'local' && !_storageListenerPaused) {
            if (changes.userSettings) {
                userSettings = {
                    ...userSettings,
                    ...(changes.userSettings.newValue || {})
                };
                updatePaidStatus();
            }
            if (changes.jobHistory) {

                // 如果当前在历史记录页面，立即刷新
                if (currentSection === 'history') {
                    loadJobHistory().then(() => {
                        filterAndDisplayJobs();
                        updateStatistics();
                        showStatus('历史记录已更新', 'success');
                    });
                }
            }

            if (changes.scoredJobs) {

                // 如果当前在历史记录页面，检查是否有新评分
                if (currentSection === 'history') {
                    checkForNewScores(changes.scoredJobs);
                }
            }
        } else if (_storageListenerPaused) {
        }
    };

    chrome.storage.onChanged.addListener(storageListener);
}

function checkForNewScores(scoredJobsChange) {
    const newScoredJobs = scoredJobsChange.newValue || {};
    const oldScoredJobs = scoredJobsChange.oldValue || {};

    // 检查是否有新的评分或更新的评分
    for (const [jobUrl, scoreData] of Object.entries(newScoredJobs)) {
        const oldScore = oldScoredJobs[jobUrl];

        // 如果是新的评分或者评分有更新
        if (!oldScore || oldScore.score !== scoreData.score) {

            // 更新对应的岗位记录
            updateJobScore(jobUrl, scoreData);
        }
    }
}

async function updateJobScore(jobUrl, scoreData) {
    try {
        // 更新本地数据
        const jobIndex = jobHistory.findIndex(job => job.jobUrl === jobUrl);
        if (jobIndex !== -1) {
            jobHistory[jobIndex].aiScore = scoreData.score;
            jobHistory[jobIndex].aiAnalysis = scoreData.aiAnalysis;
            jobHistory[jobIndex].scoredAt = scoreData.scoredAt;
            jobHistory[jobIndex].matchLevel = getMatchLevel(scoreData.score);

            // 重新渲染列表
            filterAndDisplayJobs();
            updateStatistics();

            // 如果正在显示详情弹窗，也要更新
            const modal = document.querySelector('.job-detail-modal');
            if (modal && modal.dataset.jobUrl === jobUrl) {
                updateModalWithScore(modal, jobHistory[jobIndex]);
            }
        }
    } catch (error) {
        // console.error('❌ 更新AI评分失败:', error);
    }
}

function updateModalWithScore(modal, jobData) {
    // 检查是否已存在AI评分区域
    let scoreSection = modal.querySelector('.ai-score-section');

    if (!scoreSection && jobData.aiScore) {
        // 创建AI评分区域
        scoreSection = document.createElement('div');
        scoreSection.className = 'job-detail-section ai-score-section';
        scoreSection.innerHTML = `
            <h3>AI评分结果</h3>
            <div class="score-result">
                <span class="score-value ${jobData.aiScore >= 80 ? 'high' : jobData.aiScore >= 60 ? 'medium' : 'low'}">
                    ${jobData.aiScore}分
                </span>
                <span class="match-level">${jobData.matchLevel}</span>
            </div>
            <div class="ai-analysis">
                ${parseMarkdown(jobData.aiAnalysis)}
            </div>
        `;

        // 插入到岗位详情之前
        const jobDetailSection = modal.querySelector('.job-detail-section');
        if (jobDetailSection) {
            modal.insertBefore(scoreSection, jobDetailSection);
        } else {
            modal.querySelector('.modal-body').appendChild(scoreSection);
        }
    } else if (scoreSection && jobData.aiScore) {
        // 更新现有的AI评分区域
        scoreSection.querySelector('.score-value').textContent = `${jobData.aiScore}分`;
        scoreSection.querySelector('.score-value').className = `score-value ${jobData.aiScore >= 80 ? 'high' : jobData.aiScore >= 60 ? 'medium' : 'low'}`;
        scoreSection.querySelector('.match-level').textContent = jobData.matchLevel;
        scoreSection.querySelector('.ai-analysis').innerHTML = parseMarkdown(jobData.aiAnalysis);
    }
}



// 同步BOSS简历
async function syncBossResume() {
    // 使用固定的BOSS直聘简历页面URL
    const resumeUrl = 'https://www.zhipin.com/web/geek/resume';

    try {
        // 显示加载状态
        showSyncStatus('loading');

        const response = await chrome.runtime.sendMessage({
            action: 'syncBossResume',
            resumeUrl: resumeUrl
        });

        if (response && response.success) {

            // 立即保存简历内容到storage
            try {
                await chrome.storage.local.set({ userResume: response.resumeData.content });
                userResume = response.resumeData.content;
            } catch (error) {
                // console.error('❌ 保存简历内容失败:', error);
            }

            // 更新界面
            currentResumeData = response.resumeData;
            elements.resumeContent.value = response.resumeData.content;

            // 显示成功状态
            showSyncStatus('success');

            showStatus('简历同步成功并已保存！', 'success');

        } else {
            throw new Error(response.error || '同步失败');
        }

    } catch (error) {
        // console.error('BOSS简历同步失败:', error);
        showSyncStatus('error', error.message);
        showStatus('简历同步失败：' + error.message, 'error');
    }
}

// 显示同步状态
function showSyncStatus(type, errorMessage = '') {
    const statusDiv = elements.bossSyncStatus;
    const loading = statusDiv.querySelector('.sync-loading');
    const success = statusDiv.querySelector('.sync-success');
    const error = statusDiv.querySelector('.sync-error');
    const errorMessageSpan = statusDiv.querySelector('#syncErrorMessage');

    // 隐藏所有状态
    loading.style.display = 'none';
    success.style.display = 'none';
    error.style.display = 'none';

    // 显示对应状态
    switch (type) {
        case 'loading':
            statusDiv.style.display = 'block';
            loading.style.display = 'flex';
            break;
        case 'success':
            statusDiv.style.display = 'block';
            success.style.display = 'flex';
            // 3秒后自动隐藏
            setTimeout(() => {
                statusDiv.style.display = 'none';
            }, 3000);
            break;
        case 'error':
            statusDiv.style.display = 'block';
            error.style.display = 'flex';
            if (errorMessage) {
                errorMessageSpan.textContent = errorMessage;
            }
            break;
        default:
            statusDiv.style.display = 'none';
    }
}





function showStatus(message, type = 'success') {
    // 创建（或复用）统一的提示容器，避免多条提示重叠
    let container = document.getElementById('status-toast-container');
    if (!container) {
        container = document.createElement('div');
        container.id = 'status-toast-container';
        container.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            display: flex;
            flex-direction: column;
            gap: 8px;
            align-items: flex-end;
            z-index: 10000;
        `;
        document.body.appendChild(container);
    }

    // 创建状态提示元素
    const status = document.createElement('div');
    status.className = `status-toast ${type}`;
    status.textContent = message;

    // 根据类型设置样式
    let backgroundColor = '#10b981;'; // 默认success
    if (type === 'error') {
        backgroundColor = '#ef4444;';
    } else if (type === 'warning') {
        backgroundColor = '#f59e0b;';
    } else if (type === 'loading') {
        backgroundColor = '#3b82f6;';
    }

    status.style.cssText = `
        position: relative;
        padding: 12px 20px;
        border-radius: 6px;
        color: white;
        font-size: 14px;
        animation: slideIn 0.3s ease;
        background: ${backgroundColor};
        max-width: 400px;
        word-wrap: break-word;
        box-shadow: 0 4px 10px rgba(0,0,0,0.12);
    `;

    // 如果是loading状态，添加加载图标
    if (type === 'loading') {
        const spinner = document.createElement('span');
        spinner.style.cssText = `
            display: inline-block;
            width: 16px;
            height: 16px;
            border: 2px solid rgba(255,255,255,0.3);
            border-top: 2px solid white;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 8px;
            vertical-align: middle;
        `;
        status.insertBefore(spinner, status.firstChild);
    }

    container.appendChild(status);

    // loading状态不自动移除，其他状态3秒后移除
    if (type === 'loading') {
        // 可以手动移除loading状态
        status._isLoading = true;
    } else {
        setTimeout(() => {
            if (status.parentElement === container) {
                container.removeChild(status);
            }
            if (container.childElementCount === 0 && document.body.contains(container)) {
                document.body.removeChild(container);
            }
        }, 3000);
    }
}


function cacheModelElements() {
    elements = {
        serviceCards: document.querySelectorAll('.model-card'),
        freeCurrentService: document.getElementById('freeCurrentService'),
        freeRemainingQuota: document.getElementById('freeRemainingQuota'),
        paidRemainingQuota: document.getElementById('paidRemainingQuota'),
        activatePaidServiceBtn: document.getElementById('activatePaidServiceBtn'),
        purchasePaidServiceBtn: document.getElementById('purchasePaidServiceBtn'),
        btnSelectFree: document.getElementById('btnSelectFree'),
        paidActivationSection: document.getElementById('paidActivationSection'),
        paidActivationKey: document.getElementById('paidActivationKey'),
        apiUrlInput: document.getElementById('apiUrlInput'),
        apiModelInput: document.getElementById('apiModelInput'),
        localApiKeyInput: document.getElementById('localApiKeyInput'),
        testLocalBtn: document.getElementById('testLocalBtn'),
        saveLocalBtn: document.getElementById('saveLocalBtn')
    };
}

function initializeModelMode() {
    loadLocalConfig();
    updateFreeStatus();
    updatePaidStatus();
    selectCurrentService();
}

function registerRuntimeListeners() {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
        if (message?.action === 'refreshOptions') {
            chrome.storage.local.get(['userSettings'], (result) => {
                if (result.userSettings) {
                    userSettings = {
                        ...userSettings,
                        ...result.userSettings
                    };
                }
                updatePaidStatus();
                sendResponse({ success: true });
            });
            return true;
        }
        return false;
    });
}


/**
 * 保存配置
 */

function normalizeUserInfo(userInfo) {
    const merged = {
        ...DEFAULT_CONFIG.userInfo,
        ...userInfo
    };

    if (Array.isArray(merged.blocks) && merged.blocks.length > 0) {
        return merged;
    }

    const legacyParts = [];
    if (userInfo.profile) legacyParts.push(`基础资料\n${userInfo.profile}`);
    if (userInfo.contact) legacyParts.push(`联系方式\n${userInfo.contact}`);
    if (userInfo.education) legacyParts.push(`教育经历\n${userInfo.education}`);
    if (userInfo.work) legacyParts.push(`工作经历\n${userInfo.work}`);
    if (userInfo.skills) legacyParts.push(`技能与作品\n${userInfo.skills}`);
    if (userInfo.preferences) legacyParts.push(`偏好与其他\n${userInfo.preferences}`);

    const content = legacyParts.join('\n\n').trim();
    return {
        blocks: [
            { id: 'legacy', title: '资料', content: content }
        ]
    };
}

function addProfileBlock() {
    const blocks = collectProfileBlocks();
    const newBlock = {
        id: generateId(),
        title: '资料块',
        content: ''
    };
    blocks.push(newBlock);
    renderProfileBlocks(blocks, { skipDefault: true });
}

function renderProfileBlocks(blocks, options = {}) {
    const list = document.getElementById('profileList');
    if (!list) {
        return;
    }

    list.innerHTML = '';

    const safeBlocks = Array.isArray(blocks) && blocks.length > 0
        ? blocks
        : [{ id: 'default', title: '基础资料', content: '' }];

    safeBlocks.forEach((block, index) => {
        const item = document.createElement('div');
        item.className = 'profile-item';
        item.dataset.blockId = block.id || generateId();

        const header = document.createElement('div');
        header.className = 'profile-item-header';

        const titleWrap = document.createElement('div');
        titleWrap.className = 'profile-title-wrap';

        const titleText = document.createElement('div');
        titleText.className = 'profile-title-text';
        titleText.textContent = block.title || '资料块';

        const titleInput = document.createElement('input');
        titleInput.type = 'text';
        titleInput.className = 'profile-title-input is-hidden';
        titleInput.placeholder = '资料块名称';
        titleInput.value = block.title || '';

        const actions = document.createElement('div');
        actions.className = 'profile-actions';

        const editBtn = document.createElement('button');
        editBtn.type = 'button';
        editBtn.className = 'btn-ghost btn-small';
        editBtn.textContent = '编辑名称';

        const saveBtn = document.createElement('button');
        saveBtn.type = 'button';
        saveBtn.className = 'btn-ghost btn-small is-hidden';
        saveBtn.textContent = '保存';

        const removeBtn = document.createElement('button');
        removeBtn.type = 'button';
        removeBtn.className = 'btn-danger btn-small';
        removeBtn.textContent = '删除';

        editBtn.addEventListener('click', function() {
            titleInput.classList.remove('is-hidden');
            titleText.classList.add('is-hidden');
            editBtn.classList.add('is-hidden');
            saveBtn.classList.remove('is-hidden');
            titleInput.focus();
        });

        saveBtn.addEventListener('click', function() {
            const nextTitle = titleInput.value.trim() || '资料块';
            titleText.textContent = nextTitle;
            titleInput.value = nextTitle;
            titleInput.classList.add('is-hidden');
            titleText.classList.remove('is-hidden');
            editBtn.classList.remove('is-hidden');
            saveBtn.classList.add('is-hidden');
        });

        titleInput.addEventListener('keydown', function(event) {
            if (event.key === 'Enter') {
                event.preventDefault();
                saveBtn.click();
            }
        });

        removeBtn.addEventListener('click', function() {
            removeProfileBlock(item.dataset.blockId);
        });

        titleWrap.appendChild(titleText);
        titleWrap.appendChild(titleInput);
        actions.appendChild(editBtn);
        actions.appendChild(saveBtn);
        actions.appendChild(removeBtn);
        header.appendChild(titleWrap);
        header.appendChild(actions);

        const textarea = document.createElement('textarea');
        textarea.className = 'profile-content';
        textarea.rows = 4;
        textarea.placeholder = '输入该资料块内容，建议使用条目或分行描述';
        textarea.value = block.content || '';

        item.appendChild(header);
        item.appendChild(textarea);
        list.appendChild(item);
    });

    if (!options.skipDefault && list.children.length === 0) {
        const defaultItem = document.createElement('div');
        defaultItem.className = 'profile-item';
        defaultItem.dataset.blockId = 'default';
        defaultItem.innerHTML = `
            <div class="profile-item-header">
                <div class="profile-title-wrap">
                    <div class="profile-title-text">基础资料</div>
                    <input type="text" class="profile-title-input is-hidden" placeholder="资料块名称" value="基础资料">
                </div>
                <div class="profile-actions">
                    <button type="button" class="btn-ghost btn-small">编辑名称</button>
                    <button type="button" class="btn-ghost btn-small is-hidden">保存</button>
                    <button type="button" class="btn-danger btn-small">删除</button>
                </div>
            </div>
            <textarea class="profile-content" rows="4" placeholder="输入该资料块内容，建议使用条目或分行描述"></textarea>
        `;
        list.appendChild(defaultItem);
    }
}

function removeProfileBlock(blockId) {
    const blocks = collectProfileBlocks();
    if (blocks.length <= 1) {
        showMessage('至少保留一个资料块', 'info');
        return;
    }
    const nextBlocks = blocks.filter(block => block.id !== blockId);
    renderProfileBlocks(nextBlocks, { skipDefault: true });
}

function collectProfileBlocks() {
    const list = document.getElementById('profileList');
    if (!list) {
        return [{ id: 'default', title: '基础资料', content: '' }];
    }

    const items = list.querySelectorAll('.profile-item');
    const blocks = [];

    items.forEach(item => {
        const title = item.querySelector('.profile-title-text')?.textContent?.trim() || '资料块';
        const content = item.querySelector('.profile-content')?.value?.trim() || '';
        blocks.push({
            id: item.dataset.blockId || generateId(),
            title: title,
            content: content
        });
    });

    return blocks.length > 0 ? blocks : [{ id: 'default', title: '基础资料', content: '' }];
}

function generateId() {
    return `block-${Date.now()}-${Math.random().toString(16).slice(2, 8)}`;
}
function saveConfig() {
    const config = {
        userSettings: userSettings || DEFAULT_CONFIG.userSettings,
        userInfo: {
            blocks: collectProfileBlocks()
        },
        customRules: document.getElementById('customRules').value.trim()
    };

    console.log('保存配置:', config);

    chrome.storage.local.set(config, function() {
        console.log('配置保存成功');
        showMessage('配置已保存成功！', 'success');

        // 验证保存是否成功
        chrome.storage.local.get(['userSettings'], function(result) {
            console.log('验证保存结果:', result);
        });
    });
}

/**
 * 重置配置
 */
function resetConfig() {
    if (confirm('确定要重置所有配置为默认值吗？')) {
        chrome.storage.local.set(DEFAULT_CONFIG, function() {
            loadConfig();
            showMessage('配置已重置', 'info');
        });
    }
}

/**
 * 导出配置
 */
function exportConfig() {
    chrome.storage.local.get(Object.keys(DEFAULT_CONFIG), function(result) {
        const config = { ...DEFAULT_CONFIG, ...result };

        // 不导出敏感信息
        const exportData = {
            ...config,
            userSettings: {
                ...config.userSettings,
                activationKey: config.userSettings?.activationKey ? '***已配置***' : '',
                localApiKey: config.userSettings?.localApiKey ? '***已配置***' : ''
            }
        };

        const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `autoinput-config-${new Date().toISOString().split('T')[0]}.json`;
        a.click();
        URL.revokeObjectURL(url);

        showMessage('配置已导出', 'success');
    });
}

/**
 * 导入配置
 */
function importConfig(event) {
    const file = event.target.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = function(e) {
        try {
            const config = JSON.parse(e.target.result);

            // 如果导出的配置中被标记为已配置，则保留当前密钥
            if (config.userSettings?.activationKey === '***已配置***' || config.userSettings?.localApiKey === '***已配置***') {
                chrome.storage.local.get(['userSettings'], function(result) {
                    const saved = result.userSettings || {};
                    if (config.userSettings?.activationKey === '***已配置***') {
                        config.userSettings.activationKey = saved.activationKey || '';
                    }
                    if (config.userSettings?.localApiKey === '***已配置***') {
                        config.userSettings.localApiKey = saved.localApiKey || '';
                    }
                    applyImportedConfig(config);
                });
            } else {
                applyImportedConfig(config);
            }

        } catch (error) {
            showMessage('配置文件格式错误', 'error');
        }
    };
    reader.readAsText(file);
    event.target.value = ''; // 重置文件输入
}

/**
 * 应用导入的配置
 */
function applyImportedConfig(config) {
    chrome.storage.local.set(config, function() {
        loadConfig();
        showMessage('配置已导入', 'success');
    });
}

/**
 * 测试 API 连接
 */
/**
 * 加载历史记录
 */
function loadHistory() {
    chrome.storage.local.get(['fillHistory'], function(result) {
        const historyList = document.getElementById('historyList');
        const history = result.fillHistory || [];

        if (history.length === 0) {
            historyList.innerHTML = '<li class="empty-state">暂无历史记录</li>';
            return;
        }

        historyList.innerHTML = history.map(item => `
            <li class="history-item">
                <div class="history-item-title">${escapeHtml(item.title)}</div>
                <div class="history-item-url">${escapeHtml(item.url)}</div>
                <div class="history-item-time">${formatTime(item.timestamp)}</div>
            </li>
        `).join('');
    });
}

/**
 * 显示消息
 */
function showMessage(message, type = 'info') {
    const messageBox = document.getElementById('messageBox');
    messageBox.textContent = message;
    messageBox.className = 'message show ' + type;

    setTimeout(() => {
        messageBox.classList.remove('show');
    }, 5000);
}

/**
 * 格式化时间
 */
function formatTime(timestamp) {
    const date = new Date(timestamp);
    const now = new Date();
    const diff = now - date;

    if (diff < 60000) {
        return '刚刚';
    } else if (diff < 3600000) {
        return Math.floor(diff / 60000) + ' 分钟前';
    } else if (diff < 86400000) {
        return Math.floor(diff / 3600000) + ' 小时前';
    } else {
        return date.toLocaleDateString('zh-CN');
    }
}

/**
 * HTML 转义
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
