// 智能填表助手 - popup.js

let currentFormElements = null;
let currentTabId = null;

// 初始化
document.addEventListener('DOMContentLoaded', function() {
    console.log('Popup 已加载');

    // 获取当前标签页
    chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
        currentTabId = tabs[0].id;
    });

    // 绑定按钮事件
    document.getElementById('fillBtn').addEventListener('click', fillForm);
    document.getElementById('clearBtn').addEventListener('click', clearHighlight);
    document.getElementById('settingsBtn').addEventListener('click', openSettings);
});

/**
 * AI智能填写
 */
async function fillForm() {
    // 检查API配置
    const config = await new Promise(resolve => {
        chrome.storage.local.get(['apiKey', 'apiEndpoint', 'apiModel', 'apiProvider'], resolve);
    });
    if (!config.apiKey || config.apiKey.trim() === '') {
        showResult('请先在设置中配置 API 密钥', 'error');
        return;
    }
    if (!config.apiEndpoint || !config.apiModel) {
        showResult('API 端点或模型未配置，请先在设置页填写完整', 'error');
        return;
    }

    const fillBtn = document.getElementById('fillBtn');
    const originalText = fillBtn.innerHTML;

    // 显示加载状态
    fillBtn.disabled = true;
    fillBtn.innerHTML = '<span class="loading"></span> 正在填写...';

    showResult('正在识别并调用 AI 生成填写内容...', 'info');

    try {
        const formDescription = await new Promise((resolve, reject) => {
            chrome.tabs.sendMessage(currentTabId, { action: 'identifyForm' }, function(response) {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                    return;
                }

                if (response && response.success) {
                    currentFormElements = response.formElements;
                    document.getElementById('fieldCount').textContent = response.fieldCount + ' 个';
                    resolve(response.formElements);
                } else {
                    reject(new Error('识别表单失败'));
                }
            });
        });

        // 发送消息到background处理
        const response = await chrome.runtime.sendMessage({
            action: 'identifyAndFill',
            formDescription: formDescription,
            tabId: currentTabId  // 传递tabId
        });

        if (response.success) {
            showResult('表单填写完成！', 'success');
        } else {
            showResult('填写失败：' + response.error, 'error');
        }

    } catch (error) {
        console.error('填写失败:', error);
        showResult('填写失败：' + error.message, 'error');
    } finally {
        // 恢复按钮状态
        fillBtn.disabled = false;
        fillBtn.innerHTML = originalText;
    }
}

/**
 * 清除高亮
 */
function clearHighlight() {
    chrome.tabs.sendMessage(currentTabId, { action: 'clearHighlight' }, function(response) {
        if (chrome.runtime.lastError) {
            console.error('清除高亮失败:', chrome.runtime.lastError);
            return;
        }
        showResult('已清除高亮', 'info');
    });
}

/**
 * 打开设置页面
 */
function openSettings() {
    chrome.runtime.openOptionsPage();
}

/**
 * 显示结果信息
 */
function showResult(message, type = 'info') {
    const resultBox = document.getElementById('resultBox');
    resultBox.textContent = message;
    resultBox.className = 'result-box show ' + type;

    // 3秒后自动隐藏（除非是错误）
    if (type !== 'error') {
        setTimeout(() => {
            resultBox.classList.remove('show');
        }, 5000);
    }
}

// 监听来自background的消息
chrome.runtime.onMessage.addListener(function(message, sender, sendResponse) {
    if (message.action === 'updateStatus') {
        // 可以在这里更新状态显示
    }
    sendResponse({ received: true });
});
